/* nortel_pci.c (based on orinoco_plx.c and orinoco_pci.c)
 * 
 * Driver for Prism II devices which would usually be driven by orinoco_cs,
 * but are connected to the PCI bus by a Nortel PCI-PCMCIA-Adapter. 
 *
 * Copyright (C) 2002 Tobias Hoffmann
 *           (C) 2003 Christoph Jungegger <disdos@traum404.de>
 *
 * Some of this code is borrowed from orinoco_plx.c
 *	Copyright (C) 2001 Daniel Barlow <dan@telent.net>
 * Some of this code is borrowed from orinoco_pci.c 
 *  Copyright (C) 2001 Jean Tourrilhes <jt@hpl.hp.com>
 * Some of this code is "inspired" by linux-wlan-ng-0.1.10, but nothing
 * has been copied from it. linux-wlan-ng-0.1.10 is originally :
 *	Copyright (C) 1999 AbsoluteValue Systems, Inc.  All Rights Reserved.
 * 
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License
 * at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and
 * limitations under the License.
 *
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License version 2 (the "GPL"), in
 * which case the provisions of the GPL are applicable instead of the
 * above.  If you wish to allow the use of your version of this file
 * only under the terms of the GPL and not to allow others to use your
 * version of this file under the MPL, indicate your decision by
 * deleting the provisions above and replace them with the notice and
 * other provisions required by the GPL.  If you do not delete the
 * provisions above, a recipient may use your version of this file
 * under either the MPL or the GPL.
 */

#include <linux/config.h>

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/ptrace.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/ioport.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/system.h>
#include <linux/netdevice.h>
#include <linux/if_arp.h>
#include <linux/etherdevice.h>
#include <linux/wireless.h>
#include <linux/list.h>
#include <linux/pci.h>
#include <linux/wireless.h>
#include <linux/fcntl.h>

#include <pcmcia/cisreg.h>

#include "hermes.h"
#include "orinoco.h"

static char dev_info[] = "nortel_pci";

#define COR_OFFSET    (0xe0)	/* COR attribute offset of Prism2 PC card */
#define COR_VALUE     (COR_LEVEL_REQ | COR_FUNC_ENA) /* Enable PC card with interrupt in level trigger */


/*
 * Do a soft reset of the PCI card using the Configuration Option Register
 * We need this to get going...
 * This is the part of the code that is strongly inspired from wlan-ng
 *
 * Note bis : Don't try to access HERMES_CMD during the reset phase.
 * It just won't work !
 */
static int
nortel_pci_cor_reset(struct orinoco_private *priv)
{
	unsigned long brg1_ioaddr=priv->hw.iobase1;
	unsigned long brg2_ioaddr=priv->hw.iobase2;

        /* Assert the reset until the card notice */
	outw_p(8, brg1_ioaddr+2);
	inw(brg2_ioaddr+COR_OFFSET);
	outw_p(0x80, brg2_ioaddr+COR_OFFSET);
        mdelay(1);
        printk(KERN_NOTICE "Reset done;\n");

        /* Give time for the card to recover from this hard effort */
	outw_p(0, brg2_ioaddr+COR_OFFSET);
	outw_p(0, brg2_ioaddr+COR_OFFSET);
	mdelay(1);
        printk(KERN_NOTICE "Clear Reset\n");

	/* set COR as usual*/
	outw_p(COR_VALUE, brg2_ioaddr+COR_OFFSET);
	outw_p(COR_VALUE, brg2_ioaddr+COR_OFFSET);
	mdelay(1);

	outw_p(0x228, brg1_ioaddr+2);
	
        return 0;
}

static int nortel_pci_init_one(struct pci_dev *pdev,
				const struct pci_device_id *ent)
{
	int err = 0;
	u16 *attr_mem = NULL;
	u32 reg;
	struct orinoco_private *priv = NULL;
	unsigned long brg1_ioaddr = 0, brg2_ioaddr = 0, pccard_ioaddr = 0;
	unsigned long brg1_iolen = 0, brg2_iolen = 0, pccard_iolen = 0;
	struct net_device *dev = NULL;
	int netdev_registered = 0;
	int i;

	err = pci_enable_device(pdev);
	if (err)
		return -EIO;

	brg1_ioaddr = pci_resource_start(pdev, 0);
	brg1_iolen = pci_resource_len(pdev, 0);
	if (! request_region(brg1_ioaddr, brg1_iolen, dev_info)) {
		printk(KERN_ERR "nortel_pci: Bridge I/O resource 1 0x%lx @ 0x%lx busy\n",
		       brg1_iolen, brg1_ioaddr);
		brg1_ioaddr = 0;
		err = -EBUSY;
		goto fail;
	}

	brg2_ioaddr = pci_resource_start(pdev, 1);
	brg2_iolen = pci_resource_len(pdev, 1);
	if (! request_region(brg2_ioaddr, brg2_iolen, dev_info)) {
		printk(KERN_ERR "nortel_pci: Bridge I/O resource 2 0x%lx @ 0x%lx busy\n",
		       brg2_iolen, brg2_ioaddr);
		brg2_ioaddr = 0;
		err = -EBUSY;
		goto fail;
	}

	pccard_ioaddr = pci_resource_start(pdev, 2);
	pccard_iolen = pci_resource_len(pdev, 2);
	if (! request_region(pccard_ioaddr, pccard_iolen, dev_info)) {
		printk(KERN_ERR "nortel_pci: I/O resource 0x%lx @ 0x%lx busy\n",
		       pccard_iolen, pccard_ioaddr);
		pccard_ioaddr = 0;
		err = -EBUSY;
		goto fail;
	}

	/* setup bridge */
	if (inw(brg1_ioaddr)&1) {
		printk(KERN_ERR "nortel_pci: brg1 answer1 wrong\n");
		goto fail;
	}
	outw_p(0x118,brg1_ioaddr+2);
	outw_p(0x108,brg1_ioaddr+2);
	mdelay(30);
	outw_p(0x8,brg1_ioaddr+2);
	for (i=0; i<30; i++) {
		mdelay(30);
		if (inw(brg1_ioaddr)&0x10) {
			break;
		}
        }
	if (i==30) {
		printk(KERN_ERR "nortel_pci: brg1 timed out\n");
		goto fail;
	}
	if (inw(brg2_ioaddr+0xe0)&1) {
		printk(KERN_ERR "nortel_pci: brg2 answer1 wrong\n");
		goto fail;
	}
	if (inw(brg2_ioaddr+0xe2)&1) {
		printk(KERN_ERR "nortel_pci: brg2 answer2 wrong\n");
		goto fail;
	}
	if (inw(brg2_ioaddr+0xe4)&1) {
		printk(KERN_ERR "nortel_pci: brg2 answer3 wrong\n");
		goto fail;
	}

	/* set the PCMCIA COR-Register */
	outw_p(COR_VALUE, brg2_ioaddr+COR_OFFSET);
	mdelay(1);
	reg = inw(brg2_ioaddr+COR_OFFSET);
	if (reg != COR_VALUE) {
		printk(KERN_ERR "nortel_pci: Error setting COR value (reg=%x)\n", reg);
		goto fail;
	}

	/* set leds */
	outw_p(1,brg1_ioaddr+10);


	dev = alloc_orinocodev(0, NULL);
	if (! dev) {
		err = -ENOMEM;
		goto fail;
	}

	priv = dev->priv;
	dev->base_addr = pccard_ioaddr;
	priv->hard_reset = nortel_pci_cor_reset;
	SET_MODULE_OWNER(dev);

	printk(KERN_DEBUG
	       "Detected Nortel PCI device at %s irq:%d, io addr:0x%lx\n",
	       pdev->slot_name, pdev->irq, pccard_ioaddr);

	hermes_struct_init(&(priv->hw), dev->base_addr,
			HERMES_IO, HERMES_16BIT_REGSPACING);
	pci_set_drvdata(pdev, dev);
	priv->hw.iobase1=brg1_ioaddr;
	priv->hw.iobase2=brg2_ioaddr;

	err = request_irq(pdev->irq, orinoco_interrupt, SA_SHIRQ, dev->name, dev);
	if (err) {
		printk(KERN_ERR "nortel_pci: Error allocating IRQ %d.\n", pdev->irq);
		err = -EBUSY;
		goto fail;
	}
	dev->irq = pdev->irq;
	nortel_pci_cor_reset(priv);

	err = register_netdev(dev);
	if (err) {
		printk(KERN_ERR "%s: Failed to register net device\n", dev->name);
		goto fail;
	}
	netdev_registered = 1;

	return 0;		/* succeeded */

fail:	
	if (!err) {
		err=-1;
	}
	printk(KERN_DEBUG "nortel_pci: init_one(), FAIL!\n");

	if (priv) {
		if (netdev_registered)
			unregister_netdev(dev);
		
		if (dev->irq)
			free_irq(dev->irq, dev);
		
		pci_set_drvdata(pdev, NULL);
		kfree(priv);
	}

	if (brg1_ioaddr)
		release_region(brg1_ioaddr, brg1_iolen);
	if (brg2_ioaddr)
		release_region(brg2_ioaddr, brg2_iolen);
	if (pccard_ioaddr)
		release_region(pccard_ioaddr, pccard_iolen);

	if (attr_mem)
		iounmap(attr_mem);

	pci_disable_device(pdev);

	return err;
}

static void __devexit nortel_pci_remove_one(struct pci_dev *pdev)
{
	struct net_device *dev = pci_get_drvdata(pdev);

	/* clear leds */
	/* outw_p(0,priv->hw.iobase1+10); */

	if (! dev)
		BUG();

	unregister_netdev(dev);
		
	if (dev->irq)
		free_irq(dev->irq, dev);
	
	pci_set_drvdata(pdev, NULL);
	kfree(dev);

	release_region(pci_resource_start(pdev, 0), pci_resource_len(pdev, 0));
	release_region(pci_resource_start(pdev, 1), pci_resource_len(pdev, 1));
	release_region(pci_resource_start(pdev, 2), pci_resource_len(pdev, 2));

	pci_disable_device(pdev);

}


static struct pci_device_id nortel_pci_id_table[] __devinitdata = {
	{0x126c, 0x8030, PCI_ANY_ID, PCI_ANY_ID,},	/* Nortel emobility PCI */
	{0,},
};

MODULE_DEVICE_TABLE(pci, nortel_pci_id_table);

static struct pci_driver nortel_pci_driver = {
	.name     = "nortel_pci",
	.id_table = nortel_pci_id_table,
	.probe    = nortel_pci_init_one,
	.remove   = __devexit_p(nortel_pci_remove_one),
	.suspend  = 0,
	.resume   = 0
};

static char version[] __initdata = "nortel_pci.c 0.13d (Tobias Hoffmann & Christoph Jungegger <disdos@traum404.de>)";
MODULE_AUTHOR("Christoph Jungegger <disdos@traum404.de>");
MODULE_DESCRIPTION("Driver for wireless LAN cards using the Nortel PCI bridge");
#ifdef MODULE_LICENSE
MODULE_LICENSE("Dual MPL/GPL");
#endif

static int __init nortel_pci_init(void)
{
	printk(KERN_DEBUG "%s\n", version);
	return pci_module_init(&nortel_pci_driver);
}

extern void __exit nortel_pci_exit(void)
{
	pci_unregister_driver(&nortel_pci_driver);
	current->state = TASK_UNINTERRUPTIBLE;
	schedule_timeout(HZ);
}

module_init(nortel_pci_init);
module_exit(nortel_pci_exit);

/*
 * Local variables:
 *  c-indent-level: 8
 *  c-basic-offset: 8
 *  tab-width: 8
 * End:
 */
